/***

Copyright (c) 2001, Agilent Technologies

Module Name:

    b_lxpci.c	(Best-Analyser Linux PCI Device Driver)

Abstract:

    A driver which creates (and adds to the registry) one device for each PCI
    function existing when loaded.

Notes:
  Additional functionality of each device includes Read/Write
  of I/O and Config. space.

***/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/stat.h>
#include <linux/compatmac.h>
#include <linux/devfs_fs_kernel.h>
#include <b_pci.h>
#include <b_ioctl.h>

// The majority of the code in here is only used during DriverEntry() so
// we can discard some functions after use.  We MUST declare these
// here and define them AFTER the ALLOC_PRAGMA.  (!!!!)

#if DEBUG
#define DBG_PRINT(arg...) printk(KERN_INFO arg)
#else
#define DBG_PRINT(arg...)
#endif

static struct file_operations bestpci_fops = {
    owner:      THIS_MODULE,
    read:       bestpci_read,
    write:      bestpci_write,
    ioctl:      bestpci_ioctl,
    open:       bestpci_open,
    release:    bestpci_close
};

static devfs_handle_t bestpci_dir;
static devfs_handle_t bestpci_devices[256];
static struct pci_dev *bestpci_pci_devices[256];

static struct pci_driver bestpci_driver = {
    name:       "BestPCI",
    id_table:   NULL,
    probe:      bestpci_probe,
    remove:     bestpci_remove,
};

static int numdevices=1;

#if CUSTOM_OEM1
/* Why is this global/global ? */
spinlock_t CAPI_Lock;
#endif

#if CUSTOM_OEM1
MODULE_DESCRIPTION("Agilent Linux support driver\n  made on "
				   __DATE__ " at " __TIME__);
#endif

/////////////////////////////////////////////////////////////////////////
static int __init bestpci_init_module(void) {
  // First register major-device
  int status=0;

#if CUSTOM_OEM1
  spin_lock_init(&CAPI_Lock);
#endif

  status=devfs_register_chrdev(BESTPCI_MAJOR, "bestpci", &bestpci_fops);

  if (status < 0) {
    printk(KERN_ERR "b_lxpci: unable to register char-device %d - %d\n",BESTPCI_MAJOR,status);
    return -ENODEV;
  }
  bestpci_dir=devfs_mk_dir(NULL,"best",NULL);
  if (bestpci_dir < 0) {
    printk(KERN_ERR "b_lxpci: unable to create devfs-directory\n");
  }

  bestpci_devices[0]=devfs_register(bestpci_dir, "pci0", DEVFS_FL_NONE, BESTPCI_MAJOR, 0, S_IFCHR | 00666, &bestpci_fops, NULL);
  if (bestpci_devices[0] < 0) {
    printk(KERN_ERR "b_lxpci: unable to create primary interface pci0\n");
    devfs_unregister_chrdev(BESTPCI_MAJOR,"bestpci");
  }

  status=pci_module_init(&bestpci_driver);

  if (status) {
    printk(KERN_ERR "b_lxpci: unable to register pci-driver\n");
    devfs_unregister(bestpci_devices[0]);
    devfs_unregister_chrdev(BESTPCI_MAJOR,"bestpci");
  } else {
    printk(KERN_INFO "b_lxpci: successfully loaded\n");
  }

  return status;
}

static int __devinit bestpci_probe(struct pci_dev *dev, const struct pci_device_id *id) {
    char name[7];
    int i;
    bestpci_device_info *devinfo = (bestpci_device_info *) kmalloc(sizeof(bestpci_device_info),GFP_KERNEL);

    sprintf(name,"pci%d",numdevices);
    // Create a new chardevice for the device
    bestpci_devices[numdevices]=devfs_register(bestpci_dir, name,DEVFS_FL_NONE, BESTPCI_MAJOR, numdevices, S_IFCHR | 00666, &bestpci_fops, NULL);
    if (bestpci_devices[numdevices] < 0) {
      printk(KERN_ERR "b_lxpci: unable to register char-dev %s\n",name);
      return -ENODEV;
    }

    bestpci_pci_devices[numdevices]=dev;

    devinfo->enumdevice = numdevices;

    devinfo->f_device_open = FALSE;
    devinfo->f_is_hp_best_device = FALSE;

#if CUSTOM_OEM1
    spin_lock_init(&(devinfo->lock));
#endif

    DBG_PRINT("VendorID: %x - DeviceID: %x\n",dev->vendor,dev->device);

    if (dev->vendor == HP_VENDOR_ID) {
      switch (dev->device) {
        case HP_DEVICE_2925:
        case HP_DEVICE_2926:
        case HP_DEVICE_2927:
        case HP_DEVICE_2928:
        case HP_DEVICE_2940:
            devinfo->f_is_hp_best_device=TRUE;
            devinfo->f_is_2926=TRUE;
            DBG_PRINT("is hp-best-device\n");
            break;
      }
    }
    if (dev->vendor == AGILENT_VENDOR_ID) {
      switch (dev->device) {
        case AGILENT_DEVICE_2922:
        case AGILENT_DEVICE_2929:
            devinfo->f_is_hp_best_device=TRUE;
            devinfo->f_is_2926=TRUE;
            DBG_PRINT("is agilent-best-device\n");
            break;
      }
    }

    devinfo->device_id = dev->device;
    devinfo->vendor_id = dev->vendor;
    devinfo->sub_vendor_id = dev->subsystem_vendor;
    devinfo->sub_device_id = dev->subsystem_device;
    devinfo->class = dev->class;

    devinfo->baseaddress = (void *)INVALID_BASE_IOADDR;
    devinfo->decoderstatus = DECODER_DISABLED;
    devinfo->f_using_direct_io = FALSE;

    if (devinfo->f_is_hp_best_device) {
      i=0;
      while ( (!(pci_resource_flags(dev,i)&IORESOURCE_IO)) && (i<5)) {
        i++;
      }
      if (i<5) {
        devinfo->baseaddress = ioremap(pci_resource_start(dev,i),pci_resource_len(dev,i));
        devinfo->baselen = pci_resource_len(dev,i);
        request_region((unsigned long)devinfo->baseaddress,devinfo->baselen,name);
        devinfo->f_using_direct_io = TRUE;
        devinfo->decoderstatus = DECODER_ENABLED | DECODER_LOADED;
      }
      DBG_PRINT("b_lxpci: registered io-address 0x%p length %d\n",devinfo->baseaddress,devinfo->baselen);
    }

    devinfo->f_using_direct_io = FALSE;

    devinfo->devicetype = NTIO_TYPE_PCI;
    devinfo->regwidth = 1;
    devinfo->commtimeouts = defTimeouts;
    devinfo->slot_id.u.bits.FunctionNumber = PCI_FUNC(dev->devfn);
    devinfo->slot_id.u.bits.DeviceNumber = PCI_SLOT(dev->devfn);
    devinfo->slot_id.u.bits.BusNumber = dev->bus->number;
    devinfo->slot_id.u.bits.Reserved = 0;

    pci_set_drvdata(dev,devinfo);
    
    printk(KERN_INFO "b_lxpic: registered device %s\n",name);

    numdevices++;

    return 0;
}

static __exit void bestpci_exit_module(void) {
  pci_unregister_driver(&bestpci_driver);
  devfs_unregister(bestpci_devices[0]);
  devfs_unregister_chrdev(BESTPCI_MAJOR,"bestpci");
  printk(KERN_INFO "b_lxpci: Successfully removed driver\n");
}

static int bestpci_open(struct inode *inode, struct file *file) {
  unsigned int minor = MINOR(inode->i_rdev);
  bestpci_device_info *devinfo;
  
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  if (minor>numdevices) {
    DBG_PRINT("Error: Tried to open unknown device %d\n",minor);
    return -ENODEV;
  }

  if (minor==0) {
    MOD_INC_USE_COUNT;
    return 0;
  }


  // Need to ensure we don't have competition for opening the card
#if CUSTOM_OEM1
  spin_lock_irqsave( &CAPI_Lock, flags ) ;
#endif

  devinfo=(bestpci_device_info *) pci_get_drvdata(bestpci_pci_devices[minor]);
  if (devinfo->f_device_open) {
    DBG_PRINT("Error: Device %d already open...\n",minor);
#if CUSTOM_OEM1
    spin_unlock_irqrestore( &CAPI_Lock, flags ) ;
#endif
    return -EBUSY;
  }
  
  MOD_INC_USE_COUNT;

  devinfo->f_device_open = TRUE;

#if CUSTOM_OEM1
  // Critical section for claiming a card done.
  spin_unlock_irqrestore( &CAPI_Lock, flags ) ;
#endif

  if (devinfo->f_is_hp_best_device) {
    pci_enable_device(bestpci_pci_devices[minor]);
    pci_set_master(bestpci_pci_devices[minor]);
    bestpci_disconnect(bestpci_pci_devices[minor]);
  }

  return 0;
}

static int bestpci_close(struct inode *inode, struct file *file) {
  unsigned int minor = MINOR(inode->i_rdev);
  bestpci_device_info *devinfo;

#if CUSTOM_OEM1
  unsigned long flags ;
#endif


  MOD_DEC_USE_COUNT;

  if (minor==0) {
    return 0;
  }

#if CUSTOM_OEM1
  // Need to ensure we don't have competition for open/closing the card
  spin_lock_irqsave( &CAPI_Lock, flags ) ;
#endif

  devinfo=(bestpci_device_info *) pci_get_drvdata(bestpci_pci_devices[minor]);

  devinfo->f_device_open = FALSE;
  devinfo->f_using_direct_io = FALSE;

  bestpci_disconnect(bestpci_pci_devices[minor]);

#if CUSTOM_OEM1
  spin_unlock_irqrestore( &CAPI_Lock, flags ) ;
#endif

  return 0;
}

static int bestpci_write(struct file *file, const char *buf, size_t count, loff_t *ppos) {
  unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
  bestpci_device_info *devinfo;
  struct pci_dev *dev;
  u32 data;
  size_t bytes_left;

#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (minor == 0) {
    return 0;
  } else {
    dev=bestpci_pci_devices[minor];
    devinfo=pci_get_drvdata(dev);
    bytes_left=count;
    if (devinfo->f_using_direct_io) {
      return -EINVAL; // Not yet supported
    } else {

#if CUSTOM_OEM1
      spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
      while (bytes_left) {
        pci_read_config_dword(dev,PCI_STATUS,&data);
        if (! (data & PCI_STATUS_SEND_FULL_BIT)) {
          copy_from_user(&data,buf,sizeof(u8));
          pci_write_config_byte(dev,PCI_DATA,data);
          buf += sizeof(u8);
          bytes_left -= sizeof(u8);
        } else {
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
	  return count-bytes_left;
        }
      }
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
      return count-bytes_left;
    }
  }
  return 0;
}

static int bestpci_read(struct file *file, char *buf, size_t count, loff_t *ppos) {
  unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
  bestpci_device_info *devinfo;
  struct pci_dev *dev;
  u32 data;
  size_t bytes_left;
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (minor == 0) {
    return 0;
  } else {
    dev=bestpci_pci_devices[minor];
    devinfo=pci_get_drvdata(dev);
    bytes_left=count;
    if (devinfo->f_using_direct_io) {
      return -EINVAL; // Not yet supported
    } else {
#if CUSTOM_OEM1
      spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
      while (bytes_left) {
        pci_read_config_dword(dev,PCI_STATUS,&data);
        if (data & PCI_STATUS_DATA_VALID_BIT) {
          pci_read_config_byte(dev,PCI_DATA,(u8 *)(&data));
          copy_to_user(buf,&data,sizeof(u8));
          buf += sizeof(u8);
          bytes_left -= sizeof(u8);
          data = PCI_STATUS_DATA_VALID_BIT;
          pci_write_config_dword(dev,PCI_STATUS,data);
        } else {
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
	  return count-bytes_left;
	}
      }
#if CUSTOM_OEM1
      spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
      return count-bytes_left;
    }
  }
  return 0;
}

static int bestpci_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg) {
  unsigned int minor = MINOR(inode->i_rdev);
  struct pci_dev *dev=bestpci_pci_devices[minor];
  bestpci_device_info *devinfo;
  bestpci_general_info geninfo;
  void *ptr;

  if (minor == 0) {
    if (cmd==0) {
      geninfo.version=DEV_VERSION;
      geninfo.dev_count=numdevices-1;
      copy_to_user((void *)arg,&geninfo,sizeof(geninfo));
      return 0;
    } else {
      devinfo=pci_get_drvdata(bestpci_pci_devices[cmd]);
      copy_to_user((void *)arg,devinfo,sizeof(*devinfo));
    }
    return 0;
  } else {
    devinfo=pci_get_drvdata(dev);
    switch (cmd) {
      case IOCTL_NTIO_GET_CONFIG_DW:
	  ptr=kmalloc(sizeof(u32),GFP_KERNEL);
          copy_from_user(ptr,(void *)arg,sizeof(u32));
          *((u32 *)ptr)=bestpci_read_config_dw(dev,*((u32 *)ptr));
          copy_to_user((void *)arg,ptr,sizeof(u32));
          kfree(ptr);
          break;
      case IOCTL_NTIO_SET_CONFIG_DW:
          ptr=kmalloc(sizeof(b_configdrivertype),GFP_KERNEL); 
	  copy_from_user(ptr,(void *)arg,sizeof(b_configdrivertype));
          bestpci_write_config_dw(dev,(b_configdrivertype *)ptr);
          kfree(ptr);
          break;
      case IOCTL_NTIO_READ_REG_DW:
  	  ptr=malloc(sizeof(b_accessporttype));
          copy_from_user(ptr,(void *)arg,sizeof(b_accessporttype));
          *((u32 *)ptr)=bestpci_direct_reg_read(dev,(b_accessporttype *)ptr);
          copy_to_user((void *)arg,ptr,sizeof(u32));
          kfree(ptr);
          break;
      case IOCTL_NTIO_WRITE_REG_DW:
	  ptr=malloc(sizeof(b_accessporttype));
          copy_from_user(ptr,(void *)arg,sizeof(b_accessporttype));
          bestpci_direct_reg_write(dev,(b_accessportdatatype *)arg);
          kfree(ptr);
          break;
      case IOCTL_NTIO_SET_DIRECT_IO:
          if (DECODER_READY & devinfo->decoderstatus) {
            devinfo->f_using_direct_io = TRUE;
          } else {
            return -EINVAL;
          }
          break;
      case IOCTL_NTIO_SET_HAL_IO:
          devinfo->f_using_direct_io = FALSE;
          break;
      case IOCTL_NTIO_SET_TIMEOUT:
  	  copy_from_user(&devinfo->commtimeouts, (BESTTIMEOUTS *)arg, sizeof(BESTTIMEOUTS));
          break;
      case IOCTL_NTIO_SET_REGWIDTH_PCI:
          devinfo->regwidth = arg;
          break;
      case IOCTL_NTIO_GET_BASE_ADDR:
          if (DECODER_LOADED & devinfo->decoderstatus) {
            copy_to_user((void *)arg,&devinfo->baseaddress,sizeof(u32));
          } else {
            return -EINVAL;
          }
          break;
    }
    return 0;
  }
  return -1;
}

void bestpci_disconnect(struct pci_dev *dev) {
  u32 data=0;
  bestpci_device_info *devinfo;

  devinfo=(bestpci_device_info *) pci_get_drvdata(dev);

  if (devinfo->f_is_hp_best_device) {
    pci_write_config_dword(dev,PCI_CONNECT_CMD,data);
  }
}

u32 bestpci_direct_reg_read(struct pci_dev *dev,b_accessporttype *pPortData) {
  u8 bStatus;
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  u32 data;
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (devinfo->f_using_direct_io) {
    return -1;
  } else {
#if CUSTOM_OEM1
    spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
    do {
      pci_read_config_byte(dev,ACCESSPORT_STAT,&bStatus);
    } while (!(bStatus & ACCESSPORT_STAT_PORT_RDY));

    switch (pPortData->m_cRegWidth) {
        case 1UL: data=ACCESSPORT_CMD_SIZE_BYTE;
                  break;
        case 2UL: data=ACCESSPORT_CMD_SIZE_WORD;
                  break;
        case 4UL: data=ACCESSPORT_CMD_SIZE_DWORD;
                  break;
    }
    data|=ACCESSPORT_CMD_READ;
    data|=((pPortData->m_dwAddress << 8) & 0xFFFFFF00UL);
    pci_write_config_dword(dev,ACCESSPORT_CMD_ADDRESS,data);
    do {
      pci_read_config_byte(dev,ACCESSPORT_STAT,&bStatus);
    } while (!(bStatus & ACCESSPORT_STAT_DATA_RDY));
    pci_read_config_dword(dev,ACCESSPORT_DATA,&data);

#if CUSTOM_OEM1
    spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
    return (data & (0xFFFFFFFF >> (8*(4-pPortData->m_cRegWidth))));/* fix for wrong shift value */ 
 
  }
}

void bestpci_direct_reg_write(struct pci_dev *dev,b_accessportdatatype *pPortData) {
  u8 bStatus;
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  u32 data=0;
#if CUSTOM_OEM1
  unsigned long flags ;
#endif

  // Need to avoid spin loops here on status bits -- add sleep or suspend?
  if (devinfo->f_using_direct_io) {
    return;
  } else {
#if CUSTOM_OEM1
    spin_lock_irqsave( &(devinfo->lock), flags ) ;
#endif
    do {
      pci_read_config_byte(dev,ACCESSPORT_STAT,&bStatus);
    } while (!(bStatus & ACCESSPORT_STAT_PORT_RDY));

    switch (pPortData->m_port.m_cRegWidth) {
        case 1UL: data=ACCESSPORT_CMD_SIZE_BYTE;
                  break;
        case 2UL: data=ACCESSPORT_CMD_SIZE_WORD;
                  break;
        case 4UL: data=ACCESSPORT_CMD_SIZE_DWORD;
                  break;
    }
    data&=~ACCESSPORT_CMD_READ;
    data|=((pPortData->m_port.m_dwAddress << 8) & 0xFFFFFF00UL);
    pci_write_config_dword(dev,ACCESSPORT_CMD_ADDRESS,data);
    pci_write_config_dword(dev,ACCESSPORT_DATA,pPortData->m_data);

#if CUSTOM_OEM1
    spin_unlock_irqrestore( &(devinfo->lock), flags ) ;
#endif
  }
}

u32 bestpci_read_config_dw(struct pci_dev *dev,u32 data) {
  bestpci_device_info *devinfo = pci_get_drvdata(dev);
  u32 data2;

  if (devinfo->f_using_direct_io) {
    return -1;
  } else {
    pci_read_config_dword(dev,data,&data2);
    return data2;
  }
}

void bestpci_write_config_dw(struct pci_dev *dev,b_configdrivertype *data) {
  bestpci_device_info *devinfo = pci_get_drvdata(dev);

  if (devinfo->f_using_direct_io) {
    return;
  } else {
    pci_write_config_dword(dev,data->Offset,data->Data.Data);
  }
}

static void bestpci_remove(struct pci_dev *dev) {
  bestpci_device_info *devinfo = pci_get_drvdata(dev);

  if (devinfo->baseaddress != (void *)INVALID_BASE_IOADDR) {
    DBG_PRINT("Releasing %p, devid %x, venid %x\n",devinfo->baseaddress,dev->device,dev->vendor);
    release_region((unsigned long)devinfo->baseaddress,devinfo->baselen);
    iounmap(devinfo->baseaddress);
  }
  devfs_unregister(bestpci_devices[devinfo->enumdevice]);
  kfree(devinfo);

  printk(KERN_INFO "b_lxpci: removed device %d\n",devinfo->enumdevice);
}

module_init(bestpci_init_module);
module_exit(bestpci_exit_module);
